#  
#  WordCompounder --- Executable file name: GoMusubi
#  Copyright(C) 2021 Kaoru Sagara and Syugo Nakamura 
#  This software is released under any of the GPL (see the file GPL), the LGPL(see the file LGPL), or the BSD License (see the file BSD).

from .base import Compounder


class ConsecutiveWordCompounder(Compounder):
  def __init__(self, mecab_parser):
    """
    コンストラクタ。
    
    Args:
        mecab_parser (): MeCabパーサ
    """
    super().__init__()
    self.__mecab_parser = mecab_parser
  
  def get_mecab_parser(self):
    """
    MeCabパーサを返す。

    Returns:
        mecab_parser: MeCabパーサ
    """
    return self.__mecab_parser

  def process(self):
    """
    合成語生成を実行する。
    """
    super().process()

    mecab_parser = self.get_mecab_parser()
    result_list = mecab_parser.result_list
    compound_list = self._extract_consecutive_word(result_list)

    # 優先度の高い順にソートする（文字数が多いほど優先度高とする）
    compound_list.sort(key=len, reverse=True)

    self._compound_list = compound_list
  
  def _extract_consecutive_word(self, result_list):
    """
    連続した特定品詞を合成語として生成する。

    Args:
        result_list (list): パース結果

    Returns:
        list: 生成された合成語
    """
    compound_list = []
    compound_word = ""
    compound_cnt = 0

    for parse_cnt, parse_result in enumerate(result_list):
      surface, features = parse_result

      # 連結対象かどうかを判定
      is_consecutive = False
      for consecutive_features in self.consecutive_features_list:
        is_consecutive = all([consecutive_feature in features if consecutive_feature.strip() != '*' else True for consecutive_feature in consecutive_features])
        if is_consecutive:
          # 連結条件(self.consecutive_features_list)に該当
          break

      # 連結処理
      if is_consecutive:
        compound_word += surface
        compound_cnt += 1
        
      # 切れ目の検出
      if (parse_cnt == len(result_list) - 1) or not is_consecutive:
        # 特定品詞の途切れを切れ目とし、合成語リストに追加する。
        if compound_word and compound_cnt >= 2:
          compound_list.append(compound_word)
        compound_word = ""
        compound_cnt = 0
      
    # 重複する合成語を削除する
    compound_list = list(set(compound_list))

    return compound_list
  
  @property
  def consecutive_features_list(self):
    """
    接続する品詞の指定。

    Returns:
        list: 品詞組合わせタプルのリスト
    """
    return [
      ('名詞', '*'),
      ('接頭詞', '名詞接続'),
      ('名詞', '接尾'),
      ('接頭辞', '*'),
      ('接尾辞', '名詞的'),
    ]